/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Diagnostics;

namespace Borland.Eco.Logging
{
	///<summary>
	///This class contains the categories used by the ECO logging mechanism.
	///</summary>
	public sealed class EcoTraceCategories
	{
		private EcoTraceCategories() { }
		///<summary>Category used for OCL-related events.</summary>
		public const string Ocl = "Eco.Ocl";
		///<summary>Category used for SQL-related events.</summary>
		public const string Sql = "Eco.Sql";
		///<summary>Category used for SQL meta data-related events.</summary>
		public const string SqlMeta = "Eco.SqlMeta";
		///<summary>Category used for Persistence mapper related events.</summary>
		public const string PMapper = "Eco.PMapper";
		///<summary>Category used for region-related events.</summary>
		public const string Regions = "Eco.Regions";
		///<summary>Category used for validation-related events (model, schema, data).</summary>
		public const string Validation = "Eco.Validation";
		///<summary>Function to determine if a string is one of the pre-defined categories.</summary>
		public static bool IsEcoCategory(string category)
		{
			return category == Ocl ||
					category == Sql ||
					category == PMapper ||
					category == Regions ||
					category == SqlMeta ||
					category == Validation;
		}
		private static string c_Separator = "===";
		///<summary>String constant used as separator in trace sequences.</summary>
		public static string Separator { get { return c_Separator; } set { c_Separator = value; } }
	}
	///<summary>
	///This class controls what kind of log messages are being issues by ECO.
	///</summary>
	public sealed class EcoLogSwitches
	{
		private EcoLogSwitches() { }
#if DEBUG
		static EcoLogSwitches()
		{
			// DefaultLogState is false if !DEBUG, so no need to overspecify.
			LogOcl = DefaultLogState;
			LogSql = DefaultLogState;
			LogSqlMeta = DefaultLogState;
			LogRegions = DefaultLogState;
			LogPMapper = DefaultLogState;
			LogValidation = DefaultLogState;
		}
#endif
		private static OclTraceSwitch c_OclTraceSwitch = new OclTraceSwitch(EcoTraceCategories.Ocl, "OCL invocations");
		private static SqlTraceSwitch c_SqlTraceSwitch = new SqlTraceSwitch(EcoTraceCategories.Sql, "SQL invocations");
		private static SqlMetaTraceSwitch c_SqlMetaTraceSwitch = new SqlMetaTraceSwitch(EcoTraceCategories.SqlMeta, "SQL meta manipulations");
		private static PMapperTraceSwitch c_PMapperTraceSwitch = new PMapperTraceSwitch(EcoTraceCategories.PMapper, "PMapper activities");
		private static RegionTraceSwitch c_RegionTraceSwitch = new RegionTraceSwitch(EcoTraceCategories.Regions, "Region expansion");
		private static ValidationTraceSwitch c_ValidationTraceSwitch = new ValidationTraceSwitch(EcoTraceCategories.Validation, "Validation");
		///<summary>Detemines if Ocl class messages are traced.</summary>
		public static bool LogOcl { get { return c_OclTraceSwitch.Enabled; } set { c_OclTraceSwitch.Enabled = value; } }
		///<summary>Detemines if Sql class messages are traced.</summary>
		public static bool LogSql { get { return c_SqlTraceSwitch.Enabled; } set { c_SqlTraceSwitch.Enabled = value; } }
		///<summary>Detemines if SqlMeta class messages are traced.</summary>
		public static bool LogSqlMeta { get { return c_SqlMetaTraceSwitch.Enabled; } set { c_SqlMetaTraceSwitch.Enabled = value; } }
		///<summary>Detemines if PMapper class messages are traced.</summary>
		public static bool LogPMapper { get { return c_PMapperTraceSwitch.Enabled; } set { c_PMapperTraceSwitch.Enabled = value; } }
		///<summary>Detemines if Region class messages are traced.</summary>
		public static bool LogRegions { get { return c_RegionTraceSwitch.Enabled; } set { c_RegionTraceSwitch.Enabled = value; } }
		///<summary>Detemines if Validation class messages are traced.</summary>
		public static bool LogValidation { get { return c_ValidationTraceSwitch.Enabled; } set { c_ValidationTraceSwitch.Enabled = value; } }
		///<summary>Default state for all classes of tracing.</summary>
		public static bool DefaultLogState
		{
			get
			{
#if DEBUG
				return true;
#else
				return false;
#endif
			}
		}
	}

	public class OclTraceSwitch: BooleanSwitch
	{
		internal OclTraceSwitch(string displayName, string description): base(displayName, description) {}
	}
	public class SqlTraceSwitch: BooleanSwitch
	{
		internal SqlTraceSwitch(string displayName, string description): base(displayName, description) {}
	}
	public class SqlMetaTraceSwitch: BooleanSwitch
	{
		internal SqlMetaTraceSwitch(string displayName, string description): base(displayName, description) {}
	}
	public class PMapperTraceSwitch: BooleanSwitch
	{
		internal PMapperTraceSwitch(string displayName, string description): base(displayName, description) {}
	}
	public class RegionTraceSwitch: BooleanSwitch
	{
		internal RegionTraceSwitch(string displayName, string description): base(displayName, description) {}
	}
	public class ValidationTraceSwitch: BooleanSwitch
	{
		internal ValidationTraceSwitch(string displayName, string description): base(displayName, description) {}
	}

	public class TraceEventArgs
	{
		private string m_Message;
		private string m_Category;
		private static string c_Separator = "| ";
		public TraceEventArgs(string message, string category)
		{
			m_Message = message;
			m_Category = category;
		}
		public string Message { get { return m_Message; } }
		public string Category { get { return m_Category; } }
		public override string ToString()
		{
			return Category + c_Separator + Message;
		}
	}

	public delegate void TraceLogHandler(object sender, TraceEventArgs e);

	///<summary>
	///EcoListener is a <see cref="TraceListener"/> with some additional functionality to plug itself into and out of
	///the listener collection.
	///When a <see cref="Trace"/>.Write* is captured, the category is compared with the Eco categories and if the category is indeed
	///an ECO category, the event <see cref="OnTraceLog"/> is triggered.
	///This class can - but must not necessarily - be subclassed.
	///</summary>
	public class EcoListener: TraceListener
	{
		protected EcoListener(): base() {}
		#region Control listening
		private static int m_ListenerCount;
		///<summary>
		///Makes sure <see cref="Singleton"/> is inserted into <see cref="Trace.Listeners"/>. Increases a reference count.
		///</summary>
		public static void StartListening()
		{
#if !CF
			if (!EcoListenerPresent())
				Trace.Listeners.Add(EcoListener.Singleton);
#endif
			m_ListenerCount++;
		}
		private static bool EcoListenerPresent()
		{
#if CF
			return false;
#else
			return Trace.Listeners.IndexOf(EcoListener.Singleton) > -1;
#endif
		}
		///<summary>
		///Decreases the reference count. If it reaches zero, <see cref="Singleton"/> is removed from the list of <see cref="Trace.Listeners"/>.
		///</summary>
		public static void StopListening()
		{
#if !CF
			m_ListenerCount--;
			if ((m_ListenerCount == 0) && (EcoListenerPresent()))
				Trace.Listeners.Remove(EcoListener.Singleton);
			if (m_ListenerCount < 0)
#endif
				m_ListenerCount = 0;
		}
		///<summary>
		///Sets the reference count to zero and removes <see cref="Singleton"/> from <see cref="Trace.Listeners"/>.
		///</summary>
		public static void StopAllListening()
		{
			m_ListenerCount = 1;
			StopListening();
		}
		///<summary>
		///Removes the <see cref="DefaultTraceListener"/> from <see cref="Trace.Listeners"/>.
		///</summary>
		public static void RemoveDefaultListener()
		{
#if !CF
			int defaultPosition = -1;
			foreach (TraceListener listener in Trace.Listeners)
			{
				if (listener.GetType() == typeof(DefaultTraceListener))
					defaultPosition = Trace.Listeners.IndexOf(listener);
			}
			if (defaultPosition != -1)
				Trace.Listeners.RemoveAt(defaultPosition);
#endif
		}
		#endregion
		///<summary>
		///A singleton instance of this class.
		///</summary>
		public static readonly EcoListener Singleton = new EcoListener();
		private bool m_Closed;
		///<summary>
		///Triggered when a trace-message beloning to an ECO category is intercepted.
		///</summary>
		public event TraceLogHandler OnTraceLog;
		///<summary>
		///Spawns the <see cref="OnTraceLog"/> events listeners.
		///</summary>
		protected void DoTraceMessage(string message, string category)
		{
			if (OnTraceLog != null) OnTraceLog(this, new TraceEventArgs(message, category));
		}
		///<summary>
		///If the state of the listened is <see cref="Open"/> and the category is one of the ECO categories,
		///the <see cref="OnTraceLog"/> event is triggered.
		///</summary>
		public override void Write(string message, string category)
		{
			if (!m_Closed && EcoTraceCategories.IsEcoCategory(category))
				DoTraceMessage(message, category);
		}
		///<summary>Calls <see cref="Write"/>(value.ToString(), category)</summary>
		public override void Write(object o, string category)
		{
			if (o == null) return;
			Write(o.ToString(), category);
		}
		///<summary>Empty implementation</summary>
		public override void Write(string message)
		{
			// do nothing
		}
		///<summary>Empty implementation</summary>
		public override void Write(object o)
		{
			// do nothing
		}
		///<summary>Invokes <see cref="Write"/> with a CRLF appended to the string.</summary>
		public override void WriteLine(string message, string category)
		{
			Write(message + "\n\r", category);
		}
		///<summary>Invokes <see cref="WriteLine"/> with value.ToString().</summary>
		public override void WriteLine(object o, string category)
		{
			if (o == null) return;
			WriteLine(o.ToString(), category);
		}
		///<summary>Empty implementation</summary>
		public override void WriteLine(string message)
		{
			// do nothing
		}
		///<summary>Empty implementation</summary>
		public override void WriteLine(object o)
		{
			// do nothing
		}
		///<summary>Flags the listener as closed. No events will be triggered when the listener is closed.</summary>
		public override void Close()
		{
			m_Closed = true;
		}
		///<summary>Flags the listener as open (the default state). Can be called after <see cref="Close"/> to resume listening.</summary>
		public virtual void Open()
		{
			m_Closed = false;
		}
		///<summary>Empty implementation</summary>
		public override void Flush()
		{
			// Currently nothing
		}
	}
}
